<?php
// CORS Security - Protect from unauthorized access
require_once '../../config/cors-security.php';
CorsManager::setSecureHeaders();
CorsManager::handlePreflightRequest();

// Include production config for proper error handling
require_once '../../config/production.php';

header('Content-Type: application/json');
// FIX: gunakan path absolut ke config utama
require_once __DIR__ . '/../config/database.php';
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}
$input = json_decode(file_get_contents('php://input'), true);
// Validate input
if (!$input || !isset($input['type'], $input['value'], $input['count'], $input['expiryDate'], $input['prize_name'], $input['name'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Missing required fields']);
    exit;
}
$type = $input['type'];
$value = floatval($input['value']);
$count = intval($input['count']);
$prefix = $input['prefix'] ?? '';
$expiryDate = $input['expiryDate'];
$usageLimit = intval($input['usageLimit'] ?? 1);
$prizeName = $input['prize_name']; // Nama hadiah lengkap dari prizes.html
$name = $input['name'];
$status = isset($input['status']) ? $input['status'] : 'pending';
// Validate data
if (!in_array($type, ['discount', 'cashback', 'freespin', 'bonus'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid voucher type']);
    exit;
}
if ($value < 0 || ($type !== 'freespin' && $value <= 0)) {
    echo json_encode(['success' => false, 'message' => 'Value must be greater than 0 (except for freespin)']);
    exit;
}
if ($count <= 0 || $count > 100) {
    echo json_encode(['success' => false, 'message' => 'Count must be between 1 and 100']);
    exit;
}
if (!strtotime($expiryDate) || strtotime($expiryDate) <= time()) {
    echo json_encode(['success' => false, 'message' => 'Invalid expiry date']);
    exit;
}
try {
    $pdo->beginTransaction();
    $generatedVouchers = [];
    // Prepare insert statement sesuai struktur tabel
    $stmt = $pdo->prepare("
        INSERT INTO vouchers (
            id, code, name, winner_name, type, value, prize_name, status, prefix, expiry_date, usage_limit, used_count, is_used, created_by
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    for ($i = 0; $i < $count; $i++) {
        // Generate unique voucher code
        do {
            $voucherId = uniqid() . '_' . time() . '_' . $i;
            $voucherCode = generateVoucherCode($prefix);
            // Check if code already exists
            $checkStmt = $pdo->prepare("SELECT COUNT(*) FROM vouchers WHERE code = ?");
            $checkStmt->execute([$voucherCode]);
            $exists = $checkStmt->fetchColumn() > 0;
        } while ($exists);
        // Insert voucher (14 value)
        $stmt->execute([
            $voucherId,
            $voucherCode,
            $name,
            $name, // winner_name
            $type,
            $value,
            $prizeName,
            $status, // status dari input
            $prefix,
            $expiryDate,
            $usageLimit,
            0, // used_count
            0, // is_used
            'Admin'    // created_by
        ]);
        $generatedVouchers[] = [
            'id' => $voucherId,
            'code' => $voucherCode,
            'type' => $type,
            'value' => $value,
            'prize_name' => $prizeName,
            'status' => 'pending',
            'expiryDate' => $expiryDate,
            'usageLimit' => $usageLimit,
            'usedCount' => 0,
            'used' => false,
            'createdAt' => date('Y-m-d H:i:s')
        ];
    }
    $pdo->commit();
    // Ambil kode voucher pertama (atau satu-satunya jika count=1)
    $firstCode = isset($generatedVouchers[0]['code']) ? $generatedVouchers[0]['code'] : null;
    $allCodes = array_column($generatedVouchers, 'code');
    echo json_encode([
        'success' => true,
        'message' => "Successfully generated {$count} vouchers",
        'vouchers' => $generatedVouchers,
        'code' => $firstCode,
        'codes' => $allCodes
    ]);
} catch (Exception $e) {
    $pdo->rollBack();
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Error generating vouchers: ' . $e->getMessage()
    ]);
}
function generateVoucherCode($prefix = '') {
    $timestamp = strtoupper(base_convert(time(), 10, 36));
    $random = strtoupper(substr(str_shuffle('ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789'), 0, 4));
    $code = $timestamp . $random;
    if ($prefix) {
        $code = strtoupper($prefix) . '-' . $code;
    }
    return substr($code, 0, 20); // Limit to 20 characters
}
?>
