// Initialize AOS
AOS.init({
    duration: 800,
    easing: 'ease-in-out',
    once: true,
    offset: 100
});
// Prize mapping - will be loaded dynamically
let prizeMapping = {};
let availablePrizes = [];
// Load prizes from admin panel
async function loadPrizesData() {
    try {
        // FIX: path ke hadiah
        const response = await fetch('../hadiah/get-wheel-prizes.php');
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        const data = await response.json();
        if (data.success && data.prizes) {
            availablePrizes = data.prizes;
            // Build prize mapping
            prizeMapping = {};
            availablePrizes.forEach((prize, index) => {
                const key = `prize-${index}`;
                prizeMapping[key] = prize.text;
            });
            // Update global for other scripts
            window.availablePrizes = availablePrizes;
            window.loadPrizesData = loadPrizesData;
            // Update dropdown
            updatePrizeDropdown();
            return true;
        } else {
            throw new Error('Invalid prizes data format');
        }
    } catch (error) {
        // Use fallback data
        loadFallbackPrizes();
        return false;
    }
}
// Load fallback prizes if API fails
function loadFallbackPrizes() {
    availablePrizes = [
        { text: 'JACKPOT 1JT', color: '#FF5722' },
        { text: 'Rp 100,000', color: '#FF9800' },
        { text: 'BONUS 500K', color: '#FFB74D' },
        { text: 'Voucher 50K', color: '#FFCC80' },
        { text: 'Rp 200,000', color: '#FF7043' },
        { text: 'Coba Lagi', color: '#BF360C' },
        { text: 'BONUS 888K', color: '#E64A19' },
        { text: 'Rp 300,000', color: '#FFA726' }
    ];
    // Build prize mapping
    prizeMapping = {};
    availablePrizes.forEach((prize, index) => {
        const key = `prize-${index}`;
        prizeMapping[key] = prize.text;
    });
    updatePrizeDropdown();
}
// Update prize dropdown with loaded data
function updatePrizeDropdown() {
    const prizeSelect = document.getElementById('prize-type');
    if (!prizeSelect) return;
    // Clear existing options except the first one
    prizeSelect.innerHTML = '<option value="">Pilih Hadiah dari Roda</option>';
    // Add options from loaded prizes
    availablePrizes.forEach((prize, index) => {
        const option = document.createElement('option');
        option.value = `prize-${index}`;
        option.textContent = prize.text;
        option.style.color = prize.color || '#333';
        prizeSelect.appendChild(option);
    });
}
// Generate voucher code based on prize
function generateVoucherCode(prizeValue) {
    const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    const timestamp = Date.now().toString().slice(-4); // Last 4 digits of timestamp
    // Prize code mapping for easier identification
    const prizeCodeMap = {
        'prize-0': 'JP', // JACKPOT
        'prize-1': 'RP', // Rupiah
        'prize-2': 'BN', // BONUS
        'prize-3': 'VC', // Voucher
        'prize-4': 'RP', // Rupiah
        'prize-5': 'TL', // Try Again (Coba Lagi)
        'prize-6': 'BN', // BONUS
        'prize-7': 'RP'  // Rupiah
    };
    const prizeCode = prizeCodeMap[prizeValue] || 'LW';
    let randomCode = '';
    for (let i = 0; i < 4; i++) {
        randomCode += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    // Format: [PrizeCode][Timestamp][Random] - e.g. JP123456AB
    return `${prizeCode}${timestamp}${randomCode}`;
}
// Update preview when form changes
function updatePreview() {
    const winnerName = document.getElementById('name')?.value || 'Nama Pemenang';
    const prizeType = document.getElementById('prize-type')?.value;
    const prizeText = prizeMapping[prizeType] || 'Hadiah Preview';
    const status = document.getElementById('voucher-status')?.value;
    const expiryDate = document.getElementById('expiry-date')?.value;
    const previewCard = document.querySelector('.voucher-card');
    if (previewCard) {
        previewCard.querySelector('.voucher-winner').textContent = winnerName;
        previewCard.querySelector('.voucher-prize').textContent = prizeText;
        // Generate code based on selected prize
        const previewCode = prizeType ? generateVoucherCode(prizeType) : 'SAMPLE-CODE';
        previewCard.querySelector('.voucher-code').textContent = previewCode;
        const statusElement = previewCard.querySelector('.voucher-status');
        statusElement.textContent = status ? status.toUpperCase() : '';
        statusElement.className = `voucher-status ${status}`;
        if (expiryDate) {
            const formattedDate = new Date(expiryDate).toLocaleDateString('id-ID');
            previewCard.querySelector('.voucher-expiry').textContent = `Exp: ${formattedDate}`;
        }
    }
}
// Update voucher statistics (fetch from backend)
async function updateVoucherStats() {
    try {
        const response = await fetch('get-voucher-stats.php');
        if (!response.ok) throw new Error('Gagal fetch statistik voucher');
        const data = await response.json();
        if (data.success && data.stats) {
            if(document.getElementById('total-vouchers')) document.getElementById('total-vouchers').textContent = data.stats.total;
            if(document.getElementById('active-vouchers')) document.getElementById('active-vouchers').textContent = data.stats.active;
            if(document.getElementById('used-vouchers')) document.getElementById('used-vouchers').textContent = data.stats.used;
            if(document.getElementById('expired-vouchers')) document.getElementById('expired-vouchers').textContent = data.stats.expired;
            if(document.getElementById('stat-total')) document.getElementById('stat-total').textContent = data.stats.total;
            if(document.getElementById('stat-pending')) document.getElementById('stat-pending').textContent = data.stats.pending ?? 0;
            if(document.getElementById('stat-claimed')) document.getElementById('stat-claimed').textContent = data.stats.claimed ?? 0;
            if(document.getElementById('stat-expired')) document.getElementById('stat-expired').textContent = data.stats.expired;
        }
    } catch (err) {
        if(document.getElementById('total-vouchers')) document.getElementById('total-vouchers').textContent = 0;
        if(document.getElementById('active-vouchers')) document.getElementById('active-vouchers').textContent = 0;
        if(document.getElementById('used-vouchers')) document.getElementById('used-vouchers').textContent = 0;
        if(document.getElementById('expired-vouchers')) document.getElementById('expired-vouchers').textContent = 0;
    }
}
window.updateVoucherStats = updateVoucherStats;