<?php
/**
 * PHP Configuration for Lucky Wheel Project
 * Global settings and environment configuration
 */

// Environment Configuration
define('ENV', 'development'); // development, staging, production
define('DEBUG_MODE', true);

// Application Settings
define('APP_NAME', 'Lucky Wheel');
define('APP_VERSION', '2.0.0');
define('APP_TIMEZONE', 'Asia/Jakarta');

// Set timezone
date_default_timezone_set(APP_TIMEZONE);

// Error reporting based on environment
if (ENV === 'development') {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
    ini_set('display_startup_errors', 1);
} elseif (ENV === 'production') {
    error_reporting(E_ERROR | E_WARNING | E_PARSE);
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
    ini_set('error_log', __DIR__ . '/../logs/php_errors.log');
} else {
    // staging environment
    error_reporting(E_ERROR | E_WARNING);
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
}

// Session Configuration
ini_set('session.cookie_httponly', 1);
ini_set('session.cookie_secure', 0); // Set to 1 for HTTPS
ini_set('session.use_strict_mode', 1);
ini_set('session.cookie_samesite', 'Lax');

// Cache Control
function setNoCacheHeaders() {
    header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0");
    header("Cache-Control: post-check=0, pre-check=0", false);
    header("Pragma: no-cache");
    header("Expires: " . gmdate("D, d M Y H:i:s", time() - 3600) . " GMT");
}

// Security Functions
function sanitizeInput($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    return $data;
}

function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

function validateCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Logging Function
function logActivity($message, $level = 'INFO') {
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] [$level] $message" . PHP_EOL;
    
    $logFile = __DIR__ . '/../logs/activity.log';
    file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
}

// Rate Limiting (Simple)
function checkRateLimit($identifier, $maxRequests = 60, $timeWindow = 60) {
    $key = 'rate_limit_' . md5($identifier);
    
    if (!isset($_SESSION[$key])) {
        $_SESSION[$key] = [
            'count' => 1,
            'start_time' => time()
        ];
        return true;
    }
    
    $currentTime = time();
    $timeElapsed = $currentTime - $_SESSION[$key]['start_time'];
    
    if ($timeElapsed > $timeWindow) {
        $_SESSION[$key] = [
            'count' => 1,
            'start_time' => $currentTime
        ];
        return true;
    }
    
    if ($_SESSION[$key]['count'] >= $maxRequests) {
        return false;
    }
    
    $_SESSION[$key]['count']++;
    return true;
}

// Domain Configuration for CORS
$allowedDomains = [
    'localhost',
    '127.0.0.1',
    'yourdomain.com',
    'admin.yourdomain.com',
    'www.yourdomain.com'
];

function isAllowedDomain($domain) {
    global $allowedDomains;
    return in_array($domain, $allowedDomains);
}

// Auto-create logs directory if not exists
$logsDir = __DIR__ . '/../logs';
if (!is_dir($logsDir)) {
    mkdir($logsDir, 0755, true);
    file_put_contents($logsDir . '/.htaccess', "Deny from all\n");
}

// Initialize session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Log page access
if (DEBUG_MODE) {
    $requestURI = $_SERVER['REQUEST_URI'] ?? 'unknown';
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
    $userIP = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    
    logActivity("Page access: $requestURI from IP: $userIP, User-Agent: $userAgent");
}
?>
