<?php
/**
 * CORS Security Configuration
 * Melindungi API dari akses tidak sah
 */

class CorsManager {
    private static $allowed_origins = [
        // Development
        'http://localhost',
        'http://127.0.0.1',
        
        // Production - Domain Lucky Spin
        'https://lucky-spin.ceritabahagia.lol',
        'https://www.lucky-spin.ceritabahagia.lol',
        'https://admin.lucky-spin.ceritabahagia.lol'
    ];
    
    public static function setSecureHeaders() {
        // Get origin dan informasi request
        $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
        $request_host = $_SERVER['HTTP_HOST'] ?? '';
        $referer = $_SERVER['HTTP_REFERER'] ?? '';
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        // Cek apakah ini development mode
        $is_development = (
            strpos($request_host, 'localhost') !== false ||
            strpos($request_host, '127.0.0.1') !== false ||
            strpos($request_host, '.local') !== false
        );
        
        // DETEKSI SERANGAN: Jika ada custom headers yang mencurigakan
        $suspicious_headers = [
            'X-Requested-With' => $_SERVER['HTTP_X_REQUESTED_WITH'] ?? '',
            'X-Custom-Header' => $_SERVER['HTTP_X_CUSTOM_HEADER'] ?? '',
            'Authorization' => $_SERVER['HTTP_AUTHORIZATION'] ?? ''
        ];
        
        $has_suspicious_headers = false;
        foreach ($suspicious_headers as $header => $value) {
            if (!empty($value)) {
                $has_suspicious_headers = true;
                break;
            }
        }
        
        // VALIDASI ULTRA KETAT
        if (!empty($origin)) {
            // Ada origin header - HARUS dari allowed origins
            if (!self::isAllowedOrigin($origin)) {
                self::denyAccess($origin, "CORS attack detected - Origin: $origin");
                exit();
            }
            header("Access-Control-Allow-Origin: $origin");
        } else {
            // Tidak ada origin header - periksa lebih lanjut
            if ($has_suspicious_headers) {
                // Ada headers mencurigakan tanpa origin = kemungkinan serangan
                self::denyAccess($request_host, "Suspicious request without origin header");
                exit();
            }
            
            // Request normal dari same origin
            if ($is_development) {
                // Development - pastikan dari localhost
                $valid_hosts = ['localhost', '127.0.0.1', 'localhost:80', 'localhost:8080'];
                if (!in_array($request_host, $valid_hosts)) {
                    self::denyAccess($request_host, "Invalid development host: $request_host");
                    exit();
                }
                header("Access-Control-Allow-Origin: http://localhost");
            } else {
                // Production - ULTRA KETAT
                $production_hosts = ['yourdomain.com', 'www.yourdomain.com', 'admin.yourdomain.com'];
                if (!in_array($request_host, $production_hosts)) {
                    self::denyAccess($request_host, "Invalid production host: $request_host");
                    exit();
                }
                header("Access-Control-Allow-Origin: https://$request_host");
            }
        }
        
        // Headers keamanan tambahan
        header('Access-Control-Allow-Credentials: true');
        header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
        header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With, X-CSRF-Token');
        header('Access-Control-Max-Age: 86400'); // Cache preflight 24 jam
        
        // Security headers
        header('X-Content-Type-Options: nosniff');
        header('X-Frame-Options: SAMEORIGIN');
        header('X-XSS-Protection: 1; mode=block');
        header('Referrer-Policy: strict-origin-when-cross-origin');
    }
    
    private static function isAllowedOrigin($origin) {
        if (empty($origin)) return false;
        
        // Exact match only - no wildcards or subdomain matching
        return in_array($origin, self::$allowed_origins);
    }
    
    private static function denyAccess($origin, $reason) {
        // Log untuk monitoring keamanan
        error_log(sprintf(
            "[CORS BLOCKED] %s - Origin: %s - IP: %s - User-Agent: %s",
            $reason,
            $origin,
            $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
        ));
        
        http_response_code(403);
        header('Content-Type: application/json');
        echo json_encode([
            'error' => 'Access denied',
            'message' => 'CORS policy violation',
            'code' => 'CORS_BLOCKED'
        ]);
        exit;
    }
    
    public static function handlePreflightRequest() {
        if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            // Validasi origin JUGA untuk preflight request
            $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
            
            if (!empty($origin) && !self::isAllowedOrigin($origin)) {
                // Preflight request dari origin tidak valid - BLOCK!
                self::denyAccess($origin, 'CORS preflight origin not allowed');
                exit();
            }
            
            self::setSecureHeaders();
            http_response_code(200);
            exit;
        }
    }
    
    public static function addToExistingApi($api_file_path) {
        $content = file_get_contents($api_file_path);
        
        // Cari dan ganti header CORS lama
        $old_patterns = [
            "/header\s*\(\s*['\"]Access-Control-Allow-Origin:\s*\*['\"]\s*\);?/i",
            "/header\s*\(\s*['\"]Access-Control-Allow-Origin:\s*[^'\"]*['\"]\s*\);?/i"
        ];
        
        $new_header = "require_once '../config/cors-security.php';\nCorsManager::setSecureHeaders();\nCorsManager::handlePreflightRequest();";
        
        foreach ($old_patterns as $pattern) {
            $content = preg_replace($pattern, $new_header, $content);
        }
        
        // Jika tidak ada header CORS, tambahkan di awal
        if (strpos($content, 'Access-Control') === false) {
            $content = preg_replace(
                '/(<\?php\s*)/i',
                "$1\n$new_header\n",
                $content,
                1
            );
        }
        
        file_put_contents($api_file_path, $content);
    }
}

// Auto-execute jika dipanggil langsung
if (basename($_SERVER['PHP_SELF']) !== 'cors-security.php') {
    CorsManager::setSecureHeaders();
    CorsManager::handlePreflightRequest();
}
?>
