// Prizes Management JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Initialize page
    initializePage();
    bindEvents();
    loadPrizes();
    // Update live date
    updateLiveDate();
    setInterval(updateLiveDate, 1000);
});
// Global variables
let prizes = [];
let editingPrizeId = null;
// Initialize page
function initializePage() {
    // More lenient auth check - just log status, don't redirect immediately
    const isLoggedIn = localStorage.getItem('adminLoggedIn');
    if (!isLoggedIn) {
        // Set a temporary flag to show we're checking
        localStorage.setItem('adminLoggedIn', 'checking');
    }
    // Continue with initialization
    continueInitialization();
}
// Continue initialization after auth check
function continueInitialization() {
    // Show loading bar
    showLoadingBar();
    // Hide loading bar after initialization
    setTimeout(() => {
        hideLoadingBar();
    }, 1000);
}
// Check server session - simplified
async function checkServerSession() {
    try {
        const response = await fetch('check-session.php');
        const data = await response.json();
        if (data.success && data.logged_in) {
            localStorage.setItem('adminLoggedIn', 'true');
            return true;
        } else {
            localStorage.removeItem('adminLoggedIn');
            return false;
        }
    } catch (error) {
        // Don't remove localStorage on error - could be network issue
        return localStorage.getItem('adminLoggedIn') === 'true';
    }
}
// Bind events
function bindEvents() {
    // Form submission
    document.getElementById('prize-form').addEventListener('submit', handleFormSubmit);
    // Buttons
    document.getElementById('add-prize-btn').addEventListener('click', showAddForm);
    document.getElementById('cancel-btn').addEventListener('click', resetForm);
    document.getElementById('refresh-prizes-btn').addEventListener('click', loadPrizes);
    // Quick actions
    document.getElementById('reset-probability-btn').addEventListener('click', resetProbabilities);
    document.getElementById('reorder-prizes-btn').addEventListener('click', reorderPrizes);
    document.getElementById('backup-prizes-btn').addEventListener('click', backupPrizes);
    // Form field changes
    document.getElementById('prize-type').addEventListener('change', handleTypeChange);
    document.getElementById('prize-icon').addEventListener('change', updateIconPreview);
    document.getElementById('prize-color').addEventListener('change', updateColorPreview);
    // Modal events
    document.getElementById('confirm-delete-btn').addEventListener('click', confirmDelete);
    // Close modal events
    document.querySelectorAll('[data-dismiss="modal"]').forEach(btn => {
        btn.addEventListener('click', closeModal);
    });
}
// Load prizes from server
async function loadPrizes(callback) {
    showTableLoading();
    try {
        const response = await fetch('get-prizes.php');
        if (!response.ok) {
            if (response.status === 401 || response.status === 403) {
                const sessionValid = await checkServerSession();
                if (!sessionValid) {
                    showCustomAlert('Sesi Anda telah berakhir. Silakan login kembali.', 'warning', 'Session Expired');
                    setTimeout(() => {
                        window.location.href = 'index.html';
                    }, 2000);
                    return;
                }
            }
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        const contentType = response.headers.get('content-type');
        if (!contentType || !contentType.includes('application/json')) {
            const text = await response.text();
            throw new Error('Server returned non-JSON response');
        }
        const data = await response.json();
        if (data.success) {
            prizes = data.prizes || [];
            localStorage.setItem('prizes_fallback', JSON.stringify(prizes));
            renderPrizesTable();
            updateStats();
            if (prizes.length > 0) {
                showToast('Data hadiah berhasil dimuat', 'success');
            }
            if (localStorage.getItem('adminLoggedIn') === 'checking') {
                localStorage.setItem('adminLoggedIn', 'true');
            }
            if (typeof callback === 'function') {
                callback();
            }
        } else {
            throw new Error(data.message || 'Gagal memuat data hadiah');
        }
    } catch (error) {
        showErrorFallback();
        showToast('Gagal memuat data hadiah: ' + error.message, 'error');
        const fallbackData = localStorage.getItem('prizes_fallback');
        if (fallbackData) {
            try {
                prizes = JSON.parse(fallbackData);
                renderPrizesTable();
                updateStats();
                showToast('Menggunakan data offline', 'warning');
                if (typeof callback === 'function') callback();
            } catch (e) {
                }
        }
    }
}
// Render prizes table
function renderPrizesTable() {
    const tbody = document.getElementById('prizes-table-body');
    if (prizes.length === 0) {
        tbody.innerHTML = `
            <tr class="empty-row">
                <td colspan="7">
                    <div class="empty-state">
                        <i class="fas fa-gift"></i>
                        <p>Belum ada hadiah yang ditambahkan</p>
                        <button class="btn-primary" onclick="showAddForm()">
                            <i class="fas fa-plus"></i> Tambah Hadiah
                        </button>
                    </div>
                </td>
            </tr>
        `;
        return;
    }
    // Sort by position (tanpa mengubah array global)
    const sortedPrizes = prizes.slice().sort((a, b) => a.position - b.position);
    tbody.innerHTML = sortedPrizes.map(prize => `
        <tr>
            <td>${prize.position}</td>
            <td>
                <div class="prize-name">
                    <div class="prize-icon" style="background-color: ${prize.color}">
                        <i class="${prize.icon}"></i>
                    </div>
                    ${prize.name}
                </div>
            </td>
            <td>
                <span class="prize-type-badge type-${prize.type}">
                    ${getTypeLabel(prize.type)}
                </span>
            </td>
            <td>
                <span class="prize-value">${formatPrizeValue(prize)}</span>
            </td>
            <td>
                <div class="probability-bar">
                    <div class="probability-fill">
                        <div class="probability-fill-inner" style="width: ${prize.probability}%"></div>
                    </div>
                    <span class="probability-text">${prize.probability}%</span>
                </div>
            </td>
            <td>
                <span class="status-badge status-${prize.status}">
                    ${prize.status === 'active' ? 'Aktif' : 'Nonaktif'}
                </span>
            </td>
            <td>
                <div class="action-buttons">
                    <button class="action-btn-small btn-edit" onclick="editPrize(${prize.id})" title="Edit">
                        <i class="fas fa-edit"></i>
                    </button>
                    <button class="action-btn-small btn-delete" onclick="deletePrize(${prize.id}, '${prize.name}')" title="Hapus">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </td>
        </tr>
    `).join('');
}
// Update statistics
function updateStats() {
    const totalPrizes = prizes.length;
    const activePrizes = prizes.filter(p => p.status === 'active').length;
    const totalProbability = prizes.reduce((sum, p) => sum + parseFloat(p.probability), 0);
    const totalPrizesEl = document.getElementById('total-prizes');
    if (totalPrizesEl) totalPrizesEl.textContent = totalPrizes;
    const activePrizesEl = document.getElementById('active-prizes');
    if (activePrizesEl) activePrizesEl.textContent = activePrizes;
    const totalProbabilityEl = document.getElementById('total-probability');
    if (totalProbabilityEl) totalProbabilityEl.textContent = totalProbability.toFixed(1) + '%';
}
// Form handling
function showAddForm() {
    editingPrizeId = null;
    document.getElementById('form-title').textContent = 'Tambah Hadiah Baru';
    document.getElementById('submit-text').textContent = 'Simpan Hadiah';
    resetForm();
}
function editPrize(id) {
    const prize = prizes.find(p => p.id === id);
    if (!prize) return;
    editingPrizeId = id;
    document.getElementById('form-title').textContent = 'Edit Hadiah';
    document.getElementById('submit-text').textContent = 'Update Hadiah';
    // Fill form
    document.getElementById('prize-id').value = prize.id;
    document.getElementById('prize-name').value = prize.name;
    document.getElementById('prize-type').value = prize.type;
    document.getElementById('prize-value').value = prize.value;
    document.getElementById('prize-icon').value = prize.icon;
    document.getElementById('prize-color').value = prize.color;
    document.getElementById('prize-probability').value = prize.probability;
    document.getElementById('prize-position').value = prize.position;
    document.getElementById('prize-status').value = prize.status;
}
function resetForm() {
    editingPrizeId = null;
    document.getElementById('form-title').textContent = 'Tambah Hadiah Baru';
    document.getElementById('submit-text').textContent = 'Simpan Hadiah';
    document.getElementById('prize-form').reset();
    document.getElementById('prize-color').value = '#FF6600';
    document.getElementById('prize-probability').value = '10.00';
    document.getElementById('prize-position').value = prizes.length + 1;
    document.getElementById('prize-status').value = 'active';
}
async function handleFormSubmit(e) {
    e.preventDefault();
    const formData = new FormData();
    formData.append('action', editingPrizeId ? 'update' : 'create');
    if (editingPrizeId) formData.append('id', editingPrizeId);
    formData.append('name', document.getElementById('prize-name').value);
    formData.append('type', document.getElementById('prize-type').value);
    formData.append('value', document.getElementById('prize-value').value);
    formData.append('icon', document.getElementById('prize-icon').value);
    formData.append('color', document.getElementById('prize-color').value);
    formData.append('probability', document.getElementById('prize-probability').value);
    formData.append('position', document.getElementById('prize-position').value);
    formData.append('status', document.getElementById('prize-status').value);
    try {
        const response = await fetch('manage-prizes.php', {
            method: 'POST',
            body: formData
        });
        const data = await response.json();
        if (data.success) {
            showToast(editingPrizeId ? 'Hadiah berhasil diupdate!' : 'Hadiah berhasil ditambahkan!', 'success');
            resetForm();
            loadPrizes();
        } else {
            throw new Error(data.message || 'Gagal menyimpan hadiah');
        }
    } catch (error) {
        showToast('Gagal menyimpan hadiah: ' + error.message, 'error');
    }
}
// Delete prize
function deletePrize(id, name) {
    document.getElementById('delete-prize-name').textContent = name;
    document.getElementById('confirm-delete-btn').dataset.prizeId = id;
    showModal('deleteModal');
}
async function confirmDelete() {
    const id = document.getElementById('confirm-delete-btn').dataset.prizeId;
    try {
        const formData = new FormData();
        formData.append('action', 'delete');
        formData.append('id', id);
        const response = await fetch('manage-prizes.php', {
            method: 'POST',
            body: formData
        });
        const data = await response.json();
        if (data.success) {
            showToast('Hadiah berhasil dihapus!', 'success');
            closeModal();
            loadPrizes();
        } else {
            throw new Error(data.message || 'Gagal menghapus hadiah');
        }
    } catch (error) {
        showToast('Gagal menghapus hadiah: ' + error.message, 'error');
    }
}
// Quick actions
async function resetProbabilities() {
    if (!await showCustomConfirm('Bagi rata probabilitas semua hadiah aktif?', 'Reset Probabilitas')) {
        return;
    }
    try {
        const response = await fetch('manage-prizes.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: 'action=reset_probabilities'
        });
        const data = await response.json();
        if (data.success) {
            showToast('Probabilitas berhasil diseimbangkan!', 'success');
            loadPrizes();
        } else {
            throw new Error(data.message || 'Gagal reset probabilitas');
        }
    } catch (error) {
        showToast('Gagal reset probabilitas: ' + error.message, 'error');
    }
}
async function reorderPrizes() {
    if (!await showCustomConfirm('Urutkan ulang posisi hadiah berdasarkan probabilitas?', 'Urutkan Ulang')) {
        return;
    }
    try {
        const response = await fetch('manage-prizes.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: 'action=reorder'
        });
        const data = await response.json();
        if (data.success) {
            showToast('Hadiah berhasil diurutkan ulang!', 'success');
            loadPrizes();
        } else {
            throw new Error(data.message || 'Gagal urutkan hadiah');
        }
    } catch (error) {
        showToast('Gagal urutkan hadiah: ' + error.message, 'error');
    }
}
function backupPrizes() {
    const dataStr = JSON.stringify(prizes, null, 2);
    const dataBlob = new Blob([dataStr], {type: 'application/json'});
    const url = URL.createObjectURL(dataBlob);
    const link = document.createElement('a');
    link.href = url;
    link.download = `lucky-wheel-prizes-${new Date().toISOString().split('T')[0]}.json`;
    link.click();
    URL.revokeObjectURL(url);
    showToast('Backup data berhasil didownload!', 'success');
}
// Helper functions
function getTypeLabel(type) {
    const labels = {
        money: 'Uang',
        voucher: 'Voucher',
        freespin: 'Free Spin',
        bonus: 'Bonus',
        other: 'Lainnya'
    };
    return labels[type] || type;
}
function formatPrizeValue(prize) {
    switch (prize.type) {
        case 'money':
            return 'Rp ' + parseInt(prize.value).toLocaleString('id-ID');
        case 'voucher':
            return prize.value + '%';
        case 'freespin':
            return prize.value + 'x Spin';
        case 'bonus':
            return 'Rp ' + parseInt(prize.value).toLocaleString('id-ID');
        default:
            return prize.value;
    }
}
function handleTypeChange() {
    const type = document.getElementById('prize-type').value;
    const valueInput = document.getElementById('prize-value');
    const hint = valueInput.nextElementSibling;
    switch (type) {
        case 'money':
            valueInput.placeholder = 'Contoh: 10000';
            hint.textContent = 'Masukkan nominal dalam rupiah';
            break;
        case 'voucher':
            valueInput.placeholder = 'Contoh: 20';
            hint.textContent = 'Masukkan persentase diskon';
            break;
        case 'freespin':
            valueInput.placeholder = 'Contoh: 1';
            hint.textContent = 'Jumlah free spin';
            break;
        case 'bonus':
            valueInput.placeholder = 'Contoh: 5000';
            hint.textContent = 'Nominal bonus dalam rupiah';
            break;
        default:
            valueInput.placeholder = 'Masukkan nilai';
            hint.textContent = 'Deskripsi hadiah';
    }
}
function updateIconPreview() {
    // Could add icon preview functionality here
}
function updateColorPreview() {
    // Could add color preview functionality here
}
// Loading and error states
function showTableLoading() {
    document.getElementById('prizes-table-body').innerHTML = `
        <tr class="loading-row">
            <td colspan="7">
                <div class="loading-spinner">
                    <i class="fas fa-spinner fa-spin"></i>
                    <span>Memuat data hadiah...</span>
                </div>
            </td>
        </tr>
    `;
}
function showErrorFallback() {
    document.getElementById('prizes-table-body').innerHTML = `
        <tr class="error-row">
            <td colspan="7">
                <div class="error-state">
                    <i class="fas fa-exclamation-triangle"></i>
                    <p>Gagal memuat data hadiah</p>
                    <button class="btn-primary" onclick="loadPrizes()">
                        <i class="fas fa-refresh"></i> Coba Lagi
                    </button>
                </div>
            </td>
        </tr>
    `;
}
// UI helper functions
function showLoadingBar() {
    document.querySelector('.loading-bar').style.display = 'block';
    setTimeout(() => {
        document.querySelector('.loading-progress').style.width = '100%';
    }, 100);
}
function hideLoadingBar() {
    document.querySelector('.loading-progress').style.width = '100%';
    setTimeout(() => {
        document.querySelector('.loading-bar').style.display = 'none';
        document.querySelector('.loading-progress').style.width = '0%';
    }, 500);
}
function showModal(modalId) {
    const modal = document.getElementById(modalId);
    if (!modal) {
        return;
    }
    // Cek duplikasi id confirm-delete-btn
    const allDeleteBtns = document.querySelectorAll('#confirm-delete-btn');
    if (allDeleteBtns.length > 1) {
        }
    // Pasang ulang event listener tombol hapus modal
    const confirmBtn = document.getElementById('confirm-delete-btn');
    if (confirmBtn) {
        confirmBtn.onclick = function() {
            confirmDelete();
        };
    }
    modal.style.display = 'flex';
    modal.classList.add('show');
    document.body.style.overflow = 'hidden';
}
function closeModal() {
    const modals = document.querySelectorAll('.modal');
    modals.forEach(modal => {
        modal.style.display = 'none';
        modal.classList.remove('show');
    });
    document.body.style.overflow = 'auto';
}
function updateLiveDate() {
    const now = new Date();
    const options = { 
        weekday: 'long', 
        year: 'numeric', 
        month: 'long', 
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    };
    document.getElementById('live-date').textContent = now.toLocaleDateString('id-ID', options);
}
// Toast notification system
function showToast(message, type = 'info', duration = 3000) {
    const container = document.getElementById('toast-container');
    const toast = document.createElement('div');
    toast.className = `toast toast-${type}`;
    const icon = type === 'success' ? 'check-circle' : 
                 type === 'error' ? 'times-circle' : 
                 type === 'warning' ? 'exclamation-triangle' : 'info-circle';
    toast.innerHTML = `
        <i class="fas fa-${icon}"></i>
        <span>${message}</span>
        <button class="toast-close" onclick="this.parentElement.remove()">
            <i class="fas fa-times"></i>
        </button>
    `;
    container.appendChild(toast);
    // Auto remove
    setTimeout(() => {
        if (toast.parentElement) {
            toast.remove();
        }
    }, duration);
}
// Custom alert/confirm functions (reuse from other files)
function showCustomAlert(message, type = 'info', title = 'Informasi') {
    return new Promise((resolve) => {
        const overlay = document.createElement('div');
        overlay.className = 'custom-modal-overlay';
        const modal = document.createElement('div');
        modal.className = 'custom-modal';
        const iconClass = type === 'success' ? 'fa-check-circle' : 
                         type === 'error' ? 'fa-times-circle' : 
                         type === 'warning' ? 'fa-exclamation-triangle' : 'fa-info-circle';
        modal.innerHTML = `
            <div class="modal-header ${type}">
                <i class="fas ${iconClass}"></i>
                <h3>${title}</h3>
            </div>
            <div class="modal-body">
                <p>${message}</p>
            </div>
            <div class="modal-footer">
                <button class="btn btn-primary" id="alertOkBtn">OK</button>
            </div>
        `;
        overlay.appendChild(modal);
        document.body.appendChild(overlay);
        document.getElementById('alertOkBtn').onclick = function() {
            document.body.removeChild(overlay);
            resolve(true);
        };
        overlay.onclick = function(e) {
            if (e.target === overlay) {
                document.body.removeChild(overlay);
                resolve(true);
            }
        };
    });
}
function showCustomConfirm(message, title = 'Konfirmasi') {
    return new Promise((resolve) => {
        const overlay = document.createElement('div');
        overlay.className = 'custom-modal-overlay';
        const modal = document.createElement('div');
        modal.className = 'custom-modal';
        modal.innerHTML = `
            <div class="modal-header warning">
                <i class="fas fa-question-circle"></i>
                <h3>${title}</h3>
            </div>
            <div class="modal-body">
                <p>${message}</p>
            </div>
            <div class="modal-footer">
                <button class="btn btn-secondary" id="confirmCancelBtn">Batal</button>
                <button class="btn btn-primary" id="confirmOkBtn">Ya</button>
            </div>
        `;
        overlay.appendChild(modal);
        document.body.appendChild(overlay);
        document.getElementById('confirmOkBtn').onclick = function() {
            document.body.removeChild(overlay);
            resolve(true);
        };
        document.getElementById('confirmCancelBtn').onclick = function() {
            document.body.removeChild(overlay);
            resolve(false);
        };
        overlay.onclick = function(e) {
            if (e.target === overlay) {
                document.body.removeChild(overlay);
                resolve(false);
            }
        };
    });
}
// Sudah dibersihkan: Lucky Wheel preview, test spin, dan animasi roda
// Script hanya untuk manajemen hadiah (CRUD), quick actions, form, dan utilitas admin
// Tidak ada script tidak berguna terkait Lucky Wheel preview