<?php
header('Content-Type: application/json');
require_once __DIR__ . '/../config/database.php';
// Enable CORS
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE');
header('Access-Control-Allow-Headers: Content-Type');
try {
    $action = $_POST['action'] ?? 'list';
    switch ($action) {
        case 'create':
            $result = createPrize();
            break;
        case 'update':
            $result = updatePrize();
            break;
        case 'delete':
            $result = deletePrize();
            break;
        case 'reset_probabilities':
            $result = resetProbabilities();
            break;
        case 'reorder':
            $result = reorderPrizes();
            break;
        default:
            $result = ['success' => false, 'message' => 'Invalid action'];
    }
    echo json_encode($result);
} catch (Exception $e) {
    echo json_encode([
        'success' => false,
        'message' => 'Server error: ' . $e->getMessage()
    ]);
}
function createPrize() {
    global $pdo;
    // Validate required fields
    $required = ['name', 'type', 'value', 'probability', 'position'];
    foreach ($required as $field) {
        if (!isset($_POST[$field]) || ($_POST[$field] === '' && $_POST[$field] !== '0')) {
            return ['success' => false, 'message' => "Field $field is required"];
        }
    }
    // Set default values
    $icon = $_POST['icon'] ?? 'fas fa-gift';
    $color = $_POST['color'] ?? '#FF6600';
    $status = $_POST['status'] ?? 'active';
    // Validate probability
    $probability = floatval($_POST['probability']);
    if ($probability < 0 || $probability > 100) {
        return ['success' => false, 'message' => 'Probabilitas harus antara 0-100%'];
    }
    // Check if position already exists and adjust
    $position = intval($_POST['position']);
    $stmt = $pdo->prepare("SELECT id FROM lucky_wheel_prizes WHERE position >= ? ORDER BY position");
    $stmt->execute([$position]);
    $existingPrizes = $stmt->fetchAll(PDO::FETCH_ASSOC);
    // Shift positions if needed
    if (count($existingPrizes) > 0) {
        $pdo->prepare("UPDATE lucky_wheel_prizes SET position = position + 1 WHERE position >= ?")->execute([$position]);
    }
    // Insert new prize
    $sql = "INSERT INTO lucky_wheel_prizes (name, type, value, icon, color, status, probability, position)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
    $stmt = $pdo->prepare($sql);
    if ($stmt->execute([
        $_POST['name'],
        $_POST['type'],
        $_POST['value'],
        $icon,
        $color,
        $status,
        $probability,
        $position
    ])) {
        return [
            'success' => true,
            'message' => 'Hadiah berhasil ditambahkan',
            'id' => $pdo->lastInsertId()
        ];
    } else {
        return ['success' => false, 'message' => 'Gagal menambahkan hadiah'];
    }
}
function updatePrize() {
    global $pdo;
    $id = $_POST['id'] ?? null;
    if (!$id) {
        return ['success' => false, 'message' => 'Prize ID is required'];
    }
    // Validate required fields
    $required = ['name', 'type', 'value', 'probability', 'position'];
    foreach ($required as $field) {
        if (!isset($_POST[$field]) || ($_POST[$field] === '' && $_POST[$field] !== '0')) {
            return ['success' => false, 'message' => "Field $field is required"];
        }
    }
    // Set default values
    $icon = $_POST['icon'] ?? 'fas fa-gift';
    $color = $_POST['color'] ?? '#FF6600';
    $status = $_POST['status'] ?? 'active';
    // Validate probability
    $probability = floatval($_POST['probability']);
    if ($probability < 0 || $probability > 100) {
        return ['success' => false, 'message' => 'Probabilitas harus antara 0-100%'];
    }
    // Update prize
    $sql = "UPDATE lucky_wheel_prizes
            SET name = ?, type = ?, value = ?, icon = ?, color = ?, status = ?, probability = ?, position = ?, updated_at = CURRENT_TIMESTAMP
            WHERE id = ?";
    $stmt = $pdo->prepare($sql);
    if ($stmt->execute([
        $_POST['name'],
        $_POST['type'],
        $_POST['value'],
        $icon,
        $color,
        $status,
        $probability,
        intval($_POST['position']),
        $id
    ])) {
        return [
            'success' => true,
            'message' => 'Hadiah berhasil diupdate'
        ];
    } else {
        return ['success' => false, 'message' => 'Gagal mengupdate hadiah'];
    }
}
function deletePrize() {
    global $pdo;
    $id = $_POST['id'] ?? null;
    if (!$id) {
        return ['success' => false, 'message' => 'Prize ID is required'];
    }
    // Get prize position before deleting
    $stmt = $pdo->prepare("SELECT position FROM lucky_wheel_prizes WHERE id = ?");
    $stmt->execute([$id]);
    $prize = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$prize) {
        return ['success' => false, 'message' => 'Hadiah tidak ditemukan'];
    }
    // Delete prize
    $stmt = $pdo->prepare("DELETE FROM lucky_wheel_prizes WHERE id = ?");
    if ($stmt->execute([$id])) {
        // Adjust positions of remaining prizes
        $pdo->prepare("UPDATE lucky_wheel_prizes SET position = position - 1 WHERE position > ?")->execute([$prize['position']]);
        return [
            'success' => true,
            'message' => 'Hadiah berhasil dihapus'
        ];
    } else {
        return ['success' => false, 'message' => 'Gagal menghapus hadiah'];
    }
}
function resetProbabilities() {
    global $pdo;
    // Get all active prizes
    $stmt = $pdo->query("SELECT id FROM lucky_wheel_prizes WHERE status = 'active'");
    $activePrizes = $stmt->fetchAll(PDO::FETCH_ASSOC);
    if (count($activePrizes) === 0) {
        return ['success' => false, 'message' => 'Tidak ada hadiah aktif'];
    }
    // Calculate equal probability
    $equalProbability = 100 / count($activePrizes);
    // Update all active prizes
    $stmt = $pdo->prepare("UPDATE lucky_wheel_prizes SET probability = ? WHERE status = 'active'");
    if ($stmt->execute([$equalProbability])) {
        return [
            'success' => true,
            'message' => 'Probabilitas berhasil diseimbangkan'
        ];
    } else {
        return ['success' => false, 'message' => 'Gagal reset probabilitas'];
    }
}
function reorderPrizes() {
    global $pdo;
    // Get all prizes ordered by probability (descending)
    $stmt = $pdo->query("SELECT id FROM lucky_wheel_prizes ORDER BY probability DESC, name ASC");
    $prizes = $stmt->fetchAll(PDO::FETCH_ASSOC);
    if (count($prizes) === 0) {
        return ['success' => false, 'message' => 'Tidak ada hadiah untuk diurutkan'];
    }
    // Update positions
    $position = 1;
    $stmt = $pdo->prepare("UPDATE lucky_wheel_prizes SET position = ? WHERE id = ?");
    foreach ($prizes as $prize) {
        $stmt->execute([$position, $prize['id']]);
        $position++;
    }
    return [
        'success' => true,
        'message' => 'Hadiah berhasil diurutkan ulang'
    ];
}
?>
