<?php
/**
 * Export Vouchers from Database to JSON
 * Endpoint untuk export semua voucher dari database ke format JSON
 *
 * Usage:
 * - GET /export-vouchers-db.php - Return JSON response
 * - GET /export-vouchers-db.php?download=1 - Download as file
 * - GET /export-vouchers-db.php?status=pending - Filter by status
 */
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
// Handle preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}
try {
    // Include database configuration
    require_once 'config/database.php';
    // Get query parameters
    $statusFilter = isset($_GET['status']) ? trim($_GET['status']) : '';
    $downloadFile = isset($_GET['download']) && $_GET['download'] == '1';
    $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 0;
    // Build SQL query
    $sql = "SELECT
                id,
                code,
                type,
                value,
                prize_name,
                CASE
                    WHEN is_used = 1 THEN 'used'
                    WHEN expiry_date <= NOW() THEN 'expired'
                    ELSE 'active'
                END as status,
                expiry_date as expiry,
                used_count as usage_count,
                usage_limit,
                created_at,
                CASE
                    WHEN is_used = 1 THEN created_at
                    ELSE NULL
                END as used_at,
                is_used as used
            FROM vouchers
            WHERE 1=1";
    // Add status filter if provided
    if (!empty($statusFilter)) {
        switch ($statusFilter) {
            case 'active':
                $sql .= " AND is_used = 0 AND expiry_date > NOW()";
                break;
            case 'used':
                $sql .= " AND is_used = 1";
                break;
            case 'expired':
                $sql .= " AND expiry_date <= NOW() AND is_used = 0";
                break;
        }
    }
    // Order by creation date (newest first)
    $sql .= " ORDER BY created_at DESC";
    // Prepare and execute query
    if ($limit > 0) {
        $stmt = $pdo->prepare($sql . " LIMIT ?");
        $stmt->execute([$limit]);
    } else {
        $stmt = $pdo->prepare($sql);
        $stmt->execute();
    }
    $vouchers = $stmt->fetchAll(PDO::FETCH_ASSOC);
    // Get statistics
    $statsQuery = "SELECT
                    COUNT(*) as total,
                    SUM(CASE WHEN is_used = 0 AND expiry_date > NOW() THEN 1 ELSE 0 END) as active,
                    SUM(CASE WHEN is_used = 1 THEN 1 ELSE 0 END) as used,
                    SUM(CASE WHEN expiry_date <= NOW() AND is_used = 0 THEN 1 ELSE 0 END) as expired
                   FROM vouchers";
    $statsStmt = $pdo->prepare($statsQuery);
    $statsStmt->execute();
    $stats = $statsStmt->fetch(PDO::FETCH_ASSOC);
    // Process voucher data for better format
    $processedVouchers = [];
    foreach ($vouchers as $voucher) {
        $processedVoucher = [
            'id' => (int)$voucher['id'],
            'code' => $voucher['code'],
            'type' => $voucher['type'],
            'value' => (int)$voucher['value'],
            'prize_name' => $voucher['prize_name'],
            'prizeText' => $voucher['prize_name'], // Alias for compatibility
            'status' => $voucher['status'],
            'expiry' => $voucher['expiry'],
            'usage_count' => (int)$voucher['usage_count'],
            'usage_limit' => (int)$voucher['usage_limit'],
            'maxUsage' => (int)$voucher['usage_limit'], // Alias for compatibility
            'used' => (bool)$voucher['used'],
            'created_at' => $voucher['created_at'],
            'createdAt' => $voucher['created_at'], // Alias for compatibility
            'used_at' => $voucher['used_at'],
            'usedAt' => $voucher['used_at'], // Alias for compatibility
            'is_expired' => strtotime($voucher['expiry']) < time() && $voucher['status'] !== 'used'
        ];
        $processedVouchers[] = $processedVoucher;
    }
    // Prepare response data
    $responseData = [
        'success' => true,
        'message' => 'Vouchers exported successfully',
        'export_info' => [
            'export_date' => date('Y-m-d H:i:s'),
            'export_timestamp' => time(),
            'filter_applied' => !empty($statusFilter) ? $statusFilter : 'none',
            'total_exported' => count($processedVouchers)
        ],
        'statistics' => [
            'total_vouchers' => (int)$stats['total'],
            'pending_vouchers' => (int)$stats['pending'],
            'claimed_vouchers' => (int)$stats['claimed'],
            'expired_vouchers' => (int)$stats['expired']
        ],
        'vouchers' => $processedVouchers
    ];
    // If download is requested, set appropriate headers and filename
    if ($downloadFile) {
        $filename = 'vouchers_export_' . date('Y-m-d_H-i-s');
        if (!empty($statusFilter)) {
            $filename .= '_' . $statusFilter;
        }
        $filename .= '.json';
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen(json_encode($responseData, JSON_PRETTY_PRINT)));
        echo json_encode($responseData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    } else {
        // Return JSON response
        echo json_encode($responseData, JSON_UNESCAPED_UNICODE);
    }
} catch (PDOException $e) {
    // Database error
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Database error',
        'message' => 'Failed to connect to database: ' . $e->getMessage(),
        'export_info' => [
            'export_date' => date('Y-m-d H:i:s'),
            'export_timestamp' => time(),
            'total_exported' => 0
        ],
        'vouchers' => []
    ], JSON_UNESCAPED_UNICODE);
} catch (Exception $e) {
    // General error
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Export failed',
        'message' => $e->getMessage(),
        'export_info' => [
            'export_date' => date('Y-m-d H:i:s'),
            'export_timestamp' => time(),
            'total_exported' => 0
        ],
        'vouchers' => []
    ], JSON_UNESCAPED_UNICODE);
}
?>
